import { NextRequest, NextResponse } from 'next/server';
import Stripe from 'stripe';

const stripe = new Stripe(process.env.STRIPE_SECRET_KEY!);

export async function POST(req: NextRequest) {
  try {
    const { name, email, amount, invoice, notes } = await req.json();
    if (!name || !email || !amount) {
      return NextResponse.json({ error: 'Missing required fields.' }, { status: 400 });
    }

    // Create a PaymentIntent for ACH debit
    const paymentIntent = await stripe.paymentIntents.create({
      amount: Math.round(Number(amount) * 100), // amount in cents
      currency: 'usd',
      payment_method_types: ['us_bank_account'],
      description: invoice ? `Invoice #${invoice}` : undefined,
      metadata: {
        name,
        email,
        invoice: invoice || '',
        notes: notes || '',
      },
      receipt_email: email,
    });

    return NextResponse.json({ clientSecret: paymentIntent.client_secret });
  } catch (error: unknown) {
    if (error instanceof Error) {
      console.error('Stripe error:', error);
      return NextResponse.json({ error: error.message }, { status: 500 });
    }
    return NextResponse.json({ error: 'Unknown error' }, { status: 500 });
  }
} 